function [sys,E]=EpLin(pente,m,varargin)
% Etats confins d'un objet physique dans une nergie potentielle linaire
%   >> pente    = Pente de l'nergie potentielle (J/m)
%   >> m        = Masse (par rapport  l'lectron) de l'objet physique
%   >> varargin     = Arguments d'entre optionnels
%           >> nx   = Nombre de point selon Ox (dfaut : 100)
%           >> nE   = Nombre de valeurs de l'nergie (dfaut : 1000)
%           >> Emin = Valeur minimale de l'nergie (dfaut : min(Ep))
%           >> Emax = Valeur maximale de l'nergie (dfaut : Ep(max(x)))
%
% Olivier Pujol : septembre 2013

%% Vrifications diverses

if (length(varargin)>4), error('Mauvais nombre d''arguments d''entre optionnels');end
if (nargin-length(varargin)~=2),error('Mauvais nombre d''arguments d''entre non optionnels');end
if (pente<=realmin),error('La pente est nulle');end

%% Longueur caractristique du problme et conversion de la pente en eV/nm

% Constantes
h=6.6260755E-34;    % Constante de Planck (J.s)
hbar=h/(2*pi);
eV=1.60217733E-19;  % Electron-volt (J) 
me=0.91093897E-30;  % Masse de l'lectron (kg)

Lc=(hbar^2/(2*m*me*abs(pente)))^(1/3);   % Longueur caractristique (en m)
Lc=1E9*Lc;                          % Conversion en nanomtre

pente=1E-9*pente/eV;                 % Pente J/m --> eV/nm

%% Arguments d'entre optionnels

switch length(varargin)
    case 0
        nx=100;                     % Valeur par dfaut du nombre de points selon Ox
        nE=1000;                    % Valeur par dfaut du nombre de valeurs de l'nergie
    case 1
        nx=varargin{1};             % Valeur par dfaut du nombre de points selon Ox
        nE=1000;                    % Valeur par dfaut du nombre de valeurs de l'nergie
    case 2
        nx=varargin{1};             % Valeur par dfaut du nombre de points selon Ox
        nE=varargin{2};             % Valeur par dfaut du nombre de valeurs de l'nergie
    case 4
        nx=varargin{1};             
        nE=varargin{2};             
        E=linspace(varargin{3},varargin{4},nE);     % Ensemble des valeurs de l'nergie
    otherwise
        ('Mauvais nombre d''arguments d''entre optionnels');
end

%% Energie potentielle

% Construction
x=linspace(0,10*Lc,nx);     % Domaine spatial 
Epc=pente*x;                % Energie potentielle linaire (pente en eV/nm)

% Discrtisation
[M,xt,Ep]=discretisation(x,Epc);

% Si la pente est positive, on ajoute la marche M0 en xt=0 de hauteur
% 100*Ep(end). Si la pente est ngative, on ajoute la marche M0 en
% xt=max(x) de hauteur 100*abs(Ep(end))

if (pente>0)
    M=['M0' M];                             
    xt=[0;xt];
    Ep=[100*Ep(end);Ep];
end
if (pente<0)
   M=[M 'M0'];                             
   xt=[xt;max(x)];
   Ep=[Ep;100*abs(Ep(end))];
end 

% Ensemble des valeurs de l'nergie
if (length(varargin)~=4)
    if (pente>0),E=linspace(min(Ep),Ep(end),nE);end
    if (pente<0),E=linspace(min(Ep),0,nE);end
end    

if (pente>0 && max(E)>Ep(end)),error('Certaines valeurs de E sortent du domaine des tats confins');end
if (pente<0 && max(E)>0),error('Certaines valeurs de E sortent du domaine des tats confins');end

%% Etats confins

[sys,E]=Epot(M,xt,Ep,m,E);          % Fabrication du systme
sysconf(sys,E);                     % Etats confins

return;